<?php
/**
 * Admin functionality for WooCommerce Product Feed Exporter
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class WC_Product_Feed_Admin {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_export_product_feed', array($this, 'handle_export_request'));
        add_action('admin_init', array($this, 'handle_download_request'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __('Product Feed Exporter', 'wc-product-feed-exporter'),
            __('Product Feed Export', 'wc-product-feed-exporter'),
            'manage_woocommerce',
            'wc-product-feed-exporter',
            array($this, 'admin_page_content')
        );
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_admin_scripts($hook) {
        if ('woocommerce_page_wc-product-feed-exporter' !== $hook) {
            return;
        }
        
        wp_enqueue_style(
            'wc-product-feed-exporter-admin',
            WC_PRODUCT_FEED_EXPORTER_PLUGIN_URL . 'assets/admin-styles.css',
            array(),
            WC_PRODUCT_FEED_EXPORTER_VERSION
        );
        
        wp_enqueue_script(
            'wc-product-feed-exporter-admin',
            WC_PRODUCT_FEED_EXPORTER_PLUGIN_URL . 'assets/admin-script.js',
            array('jquery'),
            WC_PRODUCT_FEED_EXPORTER_VERSION,
            true
        );
        
        wp_localize_script('wc-product-feed-exporter-admin', 'wcProductFeedExporter', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wc_product_feed_exporter_nonce'),
            'exporting_text' => __('Exporting...', 'wc-product-feed-exporter'),
            'export_complete_text' => __('Export Complete!', 'wc-product-feed-exporter'),
            'error_text' => __('Error occurred during export.', 'wc-product-feed-exporter')
        ));
    }
    
    /**
     * Admin page content
     */
    public function admin_page_content() {
        include WC_PRODUCT_FEED_EXPORTER_PLUGIN_DIR . 'admin/admin-page.php';
    }
    
    /**
     * Handle AJAX export request
     */
    public function handle_export_request() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'wc_product_feed_exporter_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Insufficient permissions');
        }
        
        try {
            $exporter = new WC_Product_Feed_CSV_Exporter();
            $file_path = $exporter->export_products();
            
            if ($file_path && file_exists($file_path)) {
                // Store file path in transient for download
                $download_key = 'wc_pfe_' . uniqid();
                set_transient($download_key, $file_path, 3600); // 1 hour expiry
                
                wp_send_json_success(array(
                    'download_url' => admin_url('admin.php?page=wc-product-feed-exporter&action=download&key=' . $download_key),
                    'message' => __('Export completed successfully!', 'wc-product-feed-exporter')
                ));
            } else {
                wp_send_json_error(__('Failed to create export file.', 'wc-product-feed-exporter'));
            }
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * Handle file download request
     */
    public function handle_download_request() {
        if (!isset($_GET['page']) || $_GET['page'] !== 'wc-product-feed-exporter') {
            return;
        }
        
        if (!isset($_GET['action']) || $_GET['action'] !== 'download') {
            return;
        }
        
        if (!isset($_GET['key'])) {
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Insufficient permissions');
        }
        
        $download_key = sanitize_text_field($_GET['key']);
        $file_path = get_transient($download_key);
        
        if (!$file_path || !file_exists($file_path)) {
            wp_die('File not found or expired.');
        }
        
        // Delete transient
        delete_transient($download_key);
        
        // Send file for download
        $filename = 'woocommerce-product-feed-' . date('Y-m-d-H-i-s') . '.csv';
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . filesize($file_path));
        header('Pragma: no-cache');
        header('Expires: 0');
        
        readfile($file_path);
        
        // Clean up temp file
        unlink($file_path);
        exit;
    }
}
