<?php
/**
 * CSV Exporter class for WooCommerce Product Feed Exporter
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class WC_Product_Feed_CSV_Exporter {
    
    /**
     * CSV headers as specified
     */
    private $csv_headers = array(
        'id',
        'title',
        'description',
        'link',
        'image_link',
        'price'
    );
    
    /**
     * Export products to CSV
     */
    public function export_products() {
        // Get all published WooCommerce products
        $products = $this->get_woocommerce_products();
        
        if (empty($products)) {
            throw new Exception(__('No products found to export.', 'wc-product-feed-exporter'));
        }
        
        // Create temporary file
        $temp_file = tempnam(sys_get_temp_dir(), 'wc_product_feed_');
        if (!$temp_file) {
            throw new Exception(__('Unable to create temporary file.', 'wc-product-feed-exporter'));
        }
        
        $handle = fopen($temp_file, 'w');
        if (!$handle) {
            throw new Exception(__('Unable to open file for writing.', 'wc-product-feed-exporter'));
        }
        
        // Write CSV headers
        fputcsv($handle, $this->csv_headers);
        
        // Write product data
        foreach ($products as $product) {
            $row_data = $this->format_product_data($product);
            fputcsv($handle, $row_data);
        }
        
        fclose($handle);
        
        return $temp_file;
    }
    
    /**
     * Get WooCommerce products
     */
    private function get_woocommerce_products() {
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => '_visibility',
                    'value' => array('catalog', 'visible'),
                    'compare' => 'IN'
                )
            )
        );
        
        $products = get_posts($args);
        $wc_products = array();
        
        foreach ($products as $product_post) {
            $product = wc_get_product($product_post->ID);
            if ($product && $product->is_visible()) {
                $wc_products[] = $product;
            }
        }
        
        return $wc_products;
    }
    
    /**
     * Format product data for CSV export
     */
    private function format_product_data($product) {
        // Get product image
        $image_id = $product->get_image_id();
        $image_url = $image_id ? wp_get_attachment_image_url($image_id, 'full') : '';
        
        // Get product description
        $description = $product->get_description();
        if (empty($description)) {
            $description = $product->get_short_description();
        }
        
        // Clean description for CSV
        $description = wp_strip_all_tags($description);
        $description = str_replace(array("\r", "\n", "\t"), ' ', $description);
        $description = trim(preg_replace('/\s+/', ' ', $description));
        
        // Get price with currency
        $price = $product->get_price();
        $currency = get_woocommerce_currency();
        $formatted_price = $price . ' ' . $currency;
        
        return array(
            $product->get_id(),                                    // id
            $product->get_name(),                                  // title
            substr($description, 0, 5000),                         // description (truncated for feed limits)
            $product->get_permalink(),                             // link
            $image_url,                                           // image_link
            $formatted_price                                      // price
        );
    }
}
